/*
    Loki Game Utility Functions
    Copyright (C) 1999  Loki Entertainment Software

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library. If not, see <https://www.gnu.org/licenses/>.
*/

#include <limits.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "loki_launch.h"

/* This function verifies that a program is in the path and executable */
static int valid_program(const char *program)
{
    char temppath[PATH_MAX];
    char *path;
    char *last;
    int found;

    found = 0;
    path = getenv("PATH");
    do {
        /* Initialize our filename variable */
        temppath[0] = '\0';

        /* Get next entry from path variable */
        last = strchr(path, ':');
        if ( ! last )
            last = path+strlen(path);

        /* Perform tilde expansion */
        if ( *path == '~' ) {
            strcpy(temppath, getenv("HOME") ? getenv("HOME") : ".");
            ++path;
        }

        /* Fill in the rest of the filename */
        if ( last > (path+1) ) {
            strncat(temppath, path, (last-path));
            strcat(temppath, "/");
        }
        strcat(temppath, "./");
        strcat(temppath, program);

        /* See if it exists, and update path */
        if ( access(temppath, X_OK) == 0 ) {
            ++found;
        }
        path = last+1;

    } while ( *last && !found );

    return found;
}

/* Replaces the string 'find' with the string 'replace' */
static const char *replace_str(const char *str, const char *find, const char *replace)
{
  static char buffer[4096];
  char *p;

  if(!(p = strstr(str, find)))  // Is 'find' even in 'str'?
    return str;

  strncpy(buffer, str, p-str); // Copy characters from 'str' start to 'find' st$
  buffer[p-str] = '\0';

  sprintf(buffer+(p-str), "%s%s", replace, p+strlen(find));

  return buffer;
}


/* This function launches the user's web browser with the given URL.
   The browser detection can be overridden by the LOKI_BROWSER environment
   variable, which is used as the format string: %s is replaced with the
   URL to be launched.
   This function returns -1 if a browser could not be found, or the return
   value of system("browser command") if one is available.
   There is no way to tell whether or not the URL was valid.

   WARNING: This function should NOT be called when a video mode is set.
 */
int loki_launchURL(const char *orig_url)
{
    /* Original website is broken, replace it with a copy that isn't */
    const char *url = replace_str(orig_url,
        "http://www.lokigames.com",
        "https://www.lokigames.twolife.be");

    /* List of programs and command strings to try */
    struct {
        char *program;
        char *command;
    } browser_list[] = {
        { "xdg-open",
          "xdg-open %s" },
        { "firefox",
          "firefox %s &" },
        { "links",
          "xterm -T \"Web Browser\" -e links %s &" },
        { "lynx",
          "xterm -T \"Web Browser\" -e lynx %s &" }
    };
    int i, status;
    char *command;
    char command_string[4*PATH_MAX];

    /* See what web browser is available */
    command = getenv("LOKI_BROWSER");
    if ( ! command || ! *command ) {
        for ( i=0; i<(sizeof browser_list)/(sizeof browser_list[0]); ++i ) {
            if ( valid_program(browser_list[i].program) ) {
                command = browser_list[i].command;
                break;
            }
        }
    }
    if ( command ) {
        sprintf(command_string, command, url, url);
        status = system(command_string);
    } else {
        status = -1;
    }
    return status;
}

void play_movie(const char *movie)
{
    /* List of programs and command strings to try */
    struct {
        char *program;
        char *command;
    } player_list[] = {
        { "mpv",
          "mpv -fs %s" },
        { "mplayer",
          "mplayer -fs %s" },
        { "ffplay",
          "ffplay -fs %s" },
        { "xdg-open",
          "xdg-open %s" }
    };
    char *command;

    for (int i=0; i<(sizeof player_list)/(sizeof player_list[0]); ++i ) {
        if ( valid_program(player_list[i].program) ) {
            command = player_list[i].command;
            break;
        }
    }

    char *command_string;
    size_t len = strlen(command) + strlen(movie);

    command_string = malloc(len);
    snprintf(command_string, len, command, movie);
    system(command_string);
    free(command_string);
}
